(* ::Subsubsection:: *)
(*CassegrainCombination package*)

(* ::Text:: *)
(*This impements the Cassegrain Combination optical system*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/CassegrainCombination"}]*)

BeginPackage["GeometricOptics`Packages`CassegrainCombination`", {"GeometricOptics`"}]

Options[CassegrainCombination] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
*)
CassegrainCombination::usage="CassegrainCombination calculates parameters for a Cassegrain Combination Camera.";

CassegrainCombinationInputPanel::usage = "CassegrainCombinationInputPanel gives the GUI panel to work with CassegrainCombination";

$CassegrainCombinationInputVariables = {"f1", "ft", "em", "diam", "\[Theta]"};

Begin["`Private`"]

CassegrainCombination[{f1_, ft_, em_, diam_, theta_}, opts___]:= CassegrainCombination[f1, ft, em, diam, theta, opts];

CassegrainCombination[f1_, ft_, em_, diam_, theta_, OptionsPattern[]] := 
 Module[{alfa, M, beta, s12, co, gamma, R2, r, D10, D11, D12, D20, D22, K11, K12, K22, K13, K23, sol, outputs, inPanel, outPanel},
 
		If[ArgumentsQ["CassegrainCombination", {$CassegrainCombinationInputVariables, {f1, ft, em, diam, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 13, "PackageName" -> "CassegrainCombination", "Arguments" -> <|"f1" -> f1, "ft" -> ft, "em" -> em, "diam" -> diam, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];

			alfa = em/f1;
			M = ft/f1;
			beta = (M - alfa)/(M + 1);
			s12 = -beta f1;
			co = 1 - beta;
			gamma = (2 M (1 + alfa))/(1 - M^2);
			R2 = gamma f1;

			(* General data section *)
			outputs = {{Style["General Data Section", Bold], ""},
						{"Primary radius", -2 f1},
						{"Secondary radius", R2},
						{"Secondary diameter", (1 - beta) diam},
						{"Distance between primary and secondary", s12},
						{"Obstruction coefficient", co}};
			
			(* Total aberrations *)
			r = diam/2;
			(* Spherical and coma aberration coefficients *)
			D10 = -((1 + M^3 + alfa - M^2 (1 + alfa))/ (M^3));
			D11 = -1;
			D12 = ((-1 + M)^3 (1 + alfa))/(M^3 (1 + M));
			D20 = (M + M^3 + alfa - M^2 alfa)/(8 f1^2 M^3);
			D22 = ((-1 + M)^3 (M - alfa))/(8 f1^2 M^3 (1 + M));
			
			K11 = D10;
			
			TotalAberrations[{-2 f1, gamma f1}, {-beta f1}, {{1, -1, 1}}, {K11, 0}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			
			(* first TotalAberration outputs - Conic primary and spherical secondary *)
			
			outputs = Join[outputs, {{Style["Conic primary and spherical secondary Section", Bold], ""},
									 {"K1", K11},
									 {"Transversal spherical Aberration", GOaberration[[1]]},
									 {"Total sagittal coma", GOcoma[[1]]},
									 {"Total astigmatism", GOastigmatism[[1]]},
									 {"Total focal", GOfocalLength[[1]]},
									 {"Height of the image", GOimageHeight[[1, GOn]]}}];
			
			(* second TotalAberration output - Spherical primary and conic secondary *)
			
			K12 = -(D10/D12);
			TotalAberrations[{-2 f1, gamma f1}, {-beta f1}, {{1, -1, 1}}, {0, K12}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			
			outputs = Join[outputs, {{Style["Spherical primary and conic secondary Section", Bold], ""},
									 {"K2", K12},
									 {"Transversal spherical Aberration", GOaberration[[1]]},
									 {"Total sagittal coma", GOcoma[[1]]},
									 {"Total Astigmatism", GOastigmatism[[1]]},
									 {"Total focal", GOfocalLength[[1]]},
									 {"Height of the image", GOimageHeight[[1, GOn]]}}];
			
			(* third TotalAberration ouput - Parabolic primary and hyperbolic secondary Section *)
			K22 = -((1 + D10)/D12);
			TotalAberrations[{-2 f1, gamma f1}, {-beta f1}, {{1, -1, 1}}, {-1, K22}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];

			outputs = Join[outputs, {{Style["Parabolic primary and hyperbolic secondary Section", Bold], ""},
									 {"K1", -1},
									 {"K2", K22},
									 {"Transversal spherical Aberration", GOaberration[[1]]},
									 {"Total sagittal coma", GOcoma[[1]]},
									 {"Total Astigmatism", GOastigmatism[[1]]},
									 {"Total focal", GOfocalLength[[1]]},
									 {"Height of the image", GOimageHeight[[1, GOn]]}}];			
			
			(* fourht TotalAberration output - Ritchey - Cretien combination Section *)
			K13 = D10 - (D12 D20)/D22;
			K23 = -(D20/D22);
			TotalAberrations[{-2 f1, gamma f1}, {-beta f1}, {{1, -1, 1}}, {K13, K23}, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]}];
			

			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Total focal length", "ft", ft},
							{"Back distance", "em", em},
							{"Diameter of primary mirror", "diam", diam},
							{"Field angle  in degree", "theta", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];
				
			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "CassegrainCombination" &], "Description"], 
												"date" -> DateString[], 
												"function" -> "CassegrainCombination", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "CassegrainCombination" &], "Description"], 
													"date" -> DateString[], 
													"function" -> "CassegrainCombination", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1" , f1},
														 {"ft", ft},
														 {"em", em},
														 {"diam", diam},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["CassegrainCombination not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

CassegrainCombinationInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "CassegrainCombination"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ft, em, diam, theta} = ReplaceAll[$CassegrainCombinationInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run "<>package<>" function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the value of focal length of the primary mirror"]},
						  {"Total focal length", "ft", Tooltip[InputField[Dynamic[ft], Alignment -> Center], "Insert the total focal length"]},
						  {"Back distance", "em", Tooltip[InputField[Dynamic[em], Alignment -> Center], "Insert the back distance"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ft, em, diam, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ft, em, diam, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]